// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_IMEDIA_BROWSER_CALLBACKS_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_IMEDIA_BROWSER_CALLBACKS_H

#include "common.h"

/**
 * Contains information about a media source.
 */
struct MediaSourceStruct {
    string path; /**< Path to this source. */
    string name; /**< Name of this source. */
    string album_art; /**< Album art of this media source as PNG. */
};

/**
 * Contains information about the root media node.
 */
struct MediaRootNodeStruct {
    string path; /**< What state the call is in. */
    struct MediaSourceStruct* sources; /**< Array of all sources available. */
    uint32_t num_sources; /**< Number of sources in array. */
};

/**
 * Contains information about a list of playable media.
 */
struct MediaListStruct {
    string path; /**< Path to this list. */
    uint32_t type; /**< The type of this list. See MediaList. */
    string name; /**< Name of this list. */
    string album_art; /**< Album art of this list as PNG. */
};

/**
 * Contains information about a source node.
 */
struct MediaSourceNodeStruct {
    struct MediaSourceStruct source; /**< Information about this source node. */
    uint32_t start; /**< Starting position of returned lists. */
    uint32_t total; /**< Total number of lists in this source. */
    struct MediaListStruct* lists; /**< Array of media lists. */
    uint32_t num_lists; /**< Number of lists in the array. */
};

/**
 * Contains information about a song.
 */
struct MediaSongStruct {
    string path; /**< Path to this song. */
    string name; /**< Name of this song. */
    string artist; /**< Artist of this song. */
    string album; /**< Name of the album. */
};

/**
 * Contains information about a list of songs.
 */
struct MediaListNodeStruct {
    struct MediaListStruct media_list; /**< Information about this media list. */
    uint32_t start; /**< Starting position of returned songs. */
    uint32_t total; /**< Total number of songs in this list. */
    struct MediaSongStruct* songs; /**< Array of songs. */
    uint32_t num_songs; /**< Number of songs in the array. */
};

/**
 * Contains information about a song with album art and duration.
 */
struct MediaSongNodeStruct {
    struct MediaSongStruct song; /**< Information about this song. */
    string album_art; /**< Album art as a PNG. */
    uint32_t duration_seconds; /**< Duration of the song in seconds. */
};

/**
 * This class represents an interface that every sink implementation must subclass if they wish to
 * implement the part of the GAL protocol allows browsing of playable media. You should also take
 * a look at the class MediaBrowserEndpoint for a pseudocode example.
 */
class IMediaBrowserCallbacks {
public:
    virtual ~IMediaBrowserCallbacks() { }

    /**
     * Called when a media root node message comes through.
     */
    virtual void rootNodeCallback(const struct MediaRootNodeStruct& node) = 0;

    /**
     * Called when a media source node message comes through.
     */
    virtual void sourceNodeCallback(const struct MediaSourceNodeStruct& node) = 0;

    /**
     * Called when a media list node message comes through.
     */
    virtual void listNodeCallback(const struct MediaListNodeStruct& node) = 0;

    /**
     * Called when a media song node message comes through.
     */
    virtual void songNodeCallback(const struct MediaSongNodeStruct& node) = 0;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_IMEDIA_BROWSER_CALLBACKS_H
